<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use App\Models\User;

class AuthController extends Controller
{
    public function showLogin()
    {
        if (auth()->check()) {
            return redirect()->intended('/');
        }
        
        return view('auth.login');
    }

    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string',
            'password' => 'required|string|min:6',
        ], [
            'phone.required' => 'رقم الهاتف مطلوب',
            'password.required' => 'كلمة المرور مطلوبة',
            'password.min' => 'كلمة المرور يجب أن تكون 6 أحرف على الأقل'
        ]);

        if ($validator->fails()) {
            return back()
                ->withErrors($validator)
                ->withInput($request->only('phone'));
        }

        $credentials = [
            'phone' => $request->phone,
            'password' => $request->password
        ];

        if (Auth::attempt($credentials, $request->filled('remember'))) {
            $request->session()->regenerate();
            
            // تحديث وقت آخر دخول
            auth()->user()->update(['last_login_at' => now()]);
            
            // إضافة نقاط ولاء لتسجيل الدخول اليومي
            if (auth()->user()->last_login_at?->isToday() === false) {
                auth()->user()->addLoyaltyPoints(5, 'تسجيل دخول يومي');
            }

            $redirectUrl = $request->get('redirect', '/');
            
            return redirect()->intended($redirectUrl)
                ->with('success', 'مرحباً بك مرة أخرى في أون فاير! 🔥');
        }

        return back()
            ->withErrors(['phone' => 'بيانات تسجيل الدخول غير صحيحة'])
            ->withInput($request->only('phone'));
    }

    public function showRegister()
    {
        if (auth()->check()) {
            return redirect('/');
        }
        
        return view('auth.register');
    }

    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'phone' => 'required|string|unique:users,phone|regex:/^01[0-2,5]{1}[0-9]{8}$/',
            'email' => 'nullable|email|unique:users,email',
            'password' => 'required|string|min:6|confirmed',
            'date_of_birth' => 'nullable|date|before:today',
            'gender' => 'nullable|in:male,female'
        ], [
            'name.required' => 'الاسم مطلوب',
            'phone.required' => 'رقم الهاتف مطلوب',
            'phone.unique' => 'رقم الهاتف مستخدم بالفعل',
            'phone.regex' => 'رقم الهاتف غير صحيح',
            'email.unique' => 'البريد الإلكتروني مستخدم بالفعل',
            'password.required' => 'كلمة المرور مطلوبة',
            'password.min' => 'كلمة المرور يجب أن تكون 6 أحرف على الأقل',
            'password.confirmed' => 'تأكيد كلمة المرور غير متطابق'
        ]);

        if ($validator->fails()) {
            return back()
                ->withErrors($validator)
                ->withInput($request->except('password', 'password_confirmation'));
        }

        $user = User::create([
            'name' => $request->name,
            'phone' => $request->phone,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'date_of_birth' => $request->date_of_birth,
            'gender' => $request->gender,
            'loyalty_points' => 50, // نقاط ترحيبية
            'membership_level' => User::LEVEL_BRONZE
        ]);

        // إضافة نقاط ولاء ترحيبية
        $user->addLoyaltyPoints(50, 'مكافأة إنشاء حساب جديد');

        Auth::login($user);

        return redirect('/')
            ->with('success', 'مرحباً بك في أون فاير! تم إنشاء حسابك بنجاح وحصلت على 50 نقطة ولاء! 🎉');
    }

    public function logout(Request $request)
    {
        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/')
            ->with('success', 'تم تسجيل الخروج بنجاح');
    }

    public function showForgotPassword()
    {
        return view('auth.forgot-password');
    }

    public function forgotPassword(Request $request)
    {
        $request->validate([
            'phone' => 'required|exists:users,phone'
        ], [
            'phone.required' => 'رقم الهاتف مطلوب',
            'phone.exists' => 'رقم الهاتف غير مسجل لدينا'
        ]);

        // في التطبيق الحقيقي، يمكن إرسال SMS هنا
        // SMS::send($request->phone, 'رمز إعادة تعيين كلمة المرور: ' . $resetCode);

        return back()->with('success', 
            'تم إرسال رسالة نصية برمز إعادة تعيين كلمة المرور إلى رقم هاتفك'
        );
    }

    public function showResetPassword($token)
    {
        return view('auth.reset-password', compact('token'));
    }

    public function resetPassword(Request $request)
    {
        $request->validate([
            'phone' => 'required|exists:users,phone',
            'code' => 'required|string',
            'password' => 'required|string|min:6|confirmed'
        ], [
            'phone.required' => 'رقم الهاتف مطلوب',
            'phone.exists' => 'رقم الهاتف غير مسجل',
            'code.required' => 'رمز التحقق مطلوب',
            'password.required' => 'كلمة المرور الجديدة مطلوبة',
            'password.min' => 'كلمة المرور يجب أن تكون 6 أحرف على الأقل',
            'password.confirmed' => 'تأكيد كلمة المرور غير متطابق'
        ]);

        // في التطبيق الحقيقي، يتم التحقق من الرمز المرسل
        // if (!$this->verifyResetCode($request->phone, $request->code)) {
        //     return back()->withErrors(['code' => 'رمز التحقق غير صحيح']);
        // }

        $user = User::where('phone', $request->phone)->first();
        $user->update([
            'password' => Hash::make($request->password)
        ]);

        Auth::login($user);

        return redirect('/')
            ->with('success', 'تم تغيير كلمة المرور بنجاح');
    }
}
