<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\CartItem;
use App\Models\Product;
use App\Models\Coupon;

class CartController extends Controller
{
    public function index()
    {
        if (auth()->check()) {
            $cartItems = auth()->user()->cartItems()->with('product')->get();
        } else {
            // للضيوف - استخدام الجلسة
            $cartItems = collect(session('cart', []));
        }

        $subtotal = $this->calculateSubtotal($cartItems);
        $deliveryFee = $this->calculateDeliveryFee($subtotal);
        $discount = session('cart_discount', 0);
        $total = $subtotal + $deliveryFee - $discount;

        return view('cart.index', compact(
            'cartItems',
            'subtotal',
            'deliveryFee', 
            'discount',
            'total'
        ));
    }

    public function add(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1|max:10',
            'customizations' => 'nullable|array',
            'special_instructions' => 'nullable|string|max:500'
        ]);

        $product = Product::findOrFail($request->product_id);
        
        if (!$product->is_active) {
            return response()->json([
                'success' => false,
                'message' => 'هذا المنتج غير متاح حالياً'
            ], 400);
        }

        if (auth()->check()) {
            // للمستخدمين المسجلين
            $existingItem = auth()->user()->cartItems()
                ->where('product_id', $product->id)
                ->first();

            if ($existingItem) {
                $existingItem->increment('quantity', $request->quantity);
                $existingItem->update([
                    'customizations' => $request->customizations,
                    'special_instructions' => $request->special_instructions
                ]);
                $item = $existingItem;
            } else {
                $item = auth()->user()->cartItems()->create([
                    'product_id' => $product->id,
                    'quantity' => $request->quantity,
                    'price' => $product->price,
                    'customizations' => $request->customizations,
                    'special_instructions' => $request->special_instructions
                ]);
            }

            // إضافة نقاط ولاء
            if (auth()->user()->getMembershipBenefits()['points_multiplier'] > 1) {
                auth()->user()->addLoyaltyPoints(1, 'إضافة منتج إلى السلة');
            }
        } else {
            // للضيوف - استخدام الجلسة
            $cart = session('cart', []);
            $itemKey = $product->id;

            if (isset($cart[$itemKey])) {
                $cart[$itemKey]['quantity'] += $request->quantity;
                $cart[$itemKey]['customizations'] = $request->customizations;
                $cart[$itemKey]['special_instructions'] = $request->special_instructions;
            } else {
                $cart[$itemKey] = [
                    'product_id' => $product->id,
                    'name' => $product->name,
                    'price' => $product->price,
                    'image' => $product->main_image,
                    'quantity' => $request->quantity,
                    'customizations' => $request->customizations,
                    'special_instructions' => $request->special_instructions
                ];
            }

            session(['cart' => $cart]);
        }

        $cartCount = $this->getCartCount();

        return response()->json([
            'success' => true,
            'message' => 'تم إضافة المنتج إلى السلة بنجاح',
            'cart_count' => $cartCount,
            'product_name' => $product->name
        ]);
    }

    public function update(Request $request, $itemId)
    {
        $request->validate([
            'quantity' => 'required|integer|min:1|max:10'
        ]);

        if (auth()->check()) {
            $cartItem = auth()->user()->cartItems()->findOrFail($itemId);
            $cartItem->update(['quantity' => $request->quantity]);
        } else {
            $cart = session('cart', []);
            if (isset($cart[$itemId])) {
                $cart[$itemId]['quantity'] = $request->quantity;
                session(['cart' => $cart]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'تم تحديث الكمية بنجاح'
        ]);
    }

    public function remove($itemId)
    {
        if (auth()->check()) {
            $cartItem = auth()->user()->cartItems()->findOrFail($itemId);
            $productName = $cartItem->product->name;
            $cartItem->delete();
        } else {
            $cart = session('cart', []);
            $productName = $cart[$itemId]['name'] ?? 'المنتج';
            unset($cart[$itemId]);
            session(['cart' => $cart]);
        }

        $cartCount = $this->getCartCount();

        return response()->json([
            'success' => true,
            'message' => "تم حذف {$productName} من السلة",
            'cart_count' => $cartCount
        ]);
    }

    public function clear()
    {
        if (auth()->check()) {
            auth()->user()->cartItems()->delete();
        } else {
            session()->forget('cart');
        }

        session()->forget('cart_discount');
        session()->forget('coupon_code');

        return response()->json([
            'success' => true,
            'message' => 'تم مسح السلة بالكامل',
            'cart_count' => 0
        ]);
    }

    public function applyCoupon(Request $request)
    {
        $request->validate([
            'coupon_code' => 'required|string'
        ]);

        $coupon = Coupon::where('code', $request->coupon_code)
            ->where('is_active', true)
            ->where('starts_at', '<=', now())
            ->where('expires_at', '>=', now())
            ->first();

        if (!$coupon) {
            return response()->json([
                'success' => false,
                'message' => 'كود الخصم غير صالح أو منتهي الصلاحية'
            ], 400);
        }

        // التحقق من الحد الأدنى للطلب
        $subtotal = $this->calculateSubtotal();
        if ($coupon->minimum_amount && $subtotal < $coupon->minimum_amount) {
            return response()->json([
                'success' => false,
                'message' => "الحد الأدنى للطلب {$coupon->minimum_amount} جنيه"
            ], 400);
        }

        // التحقق من عدد مرات الاستخدام
        if ($coupon->usage_limit && $coupon->used_count >= $coupon->usage_limit) {
            return response()->json([
                'success' => false,
                'message' => 'تم استنفاد عدد مرات استخدام هذا الكود'
            ], 400);
        }

        // حساب قيمة الخصم
        $discountAmount = 0;
        if ($coupon->type === 'percentage') {
            $discountAmount = ($subtotal * $coupon->value) / 100;
            if ($coupon->max_discount_amount) {
                $discountAmount = min($discountAmount, $coupon->max_discount_amount);
            }
        } else {
            $discountAmount = $coupon->value;
        }

        session([
            'cart_discount' => $discountAmount,
            'coupon_code' => $coupon->code,
            'coupon_id' => $coupon->id
        ]);

        return response()->json([
            'success' => true,
            'message' => 'تم تطبيق كود الخصم بنجاح',
            'discount_amount' => $discountAmount,
            'coupon_code' => $coupon->code
        ]);
    }

    public function removeCoupon()
    {
        session()->forget(['cart_discount', 'coupon_code', 'coupon_id']);

        return response()->json([
            'success' => true,
            'message' => 'تم إلغاء كود الخصم'
        ]);
    }

    public function getCount()
    {
        $count = $this->getCartCount();
        
        return response()->json(['count' => $count]);
    }

    private function getCartCount()
    {
        if (auth()->check()) {
            return auth()->user()->cartItems()->sum('quantity');
        } else {
            $cart = session('cart', []);
            return collect($cart)->sum('quantity');
        }
    }

    private function calculateSubtotal($cartItems = null)
    {
        if (!$cartItems) {
            if (auth()->check()) {
                $cartItems = auth()->user()->cartItems()->with('product')->get();
            } else {
                $cartItems = collect(session('cart', []));
            }
        }

        $subtotal = 0;
        
        if (auth()->check()) {
            foreach ($cartItems as $item) {
                $subtotal += $item->price * $item->quantity;
            }
        } else {
            foreach ($cartItems as $item) {
                $subtotal += $item['price'] * $item['quantity'];
            }
        }

        return $subtotal;
    }

    private function calculateDeliveryFee($subtotal)
    {
        $deliveryFee = env('DELIVERY_FEE', 20);
        $freeDeliveryThreshold = env('FREE_DELIVERY_THRESHOLD', 150);

        // التوصيل المجاني للعضويات المميزة
        if (auth()->check() && auth()->user()->isEligibleForFreeDelivery()) {
            return 0;
        }

        // التوصيل المجاني عند الوصول للحد الأدنى
        if ($subtotal >= $freeDeliveryThreshold) {
            return 0;
        }

        return $deliveryFee;
    }
}
