<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Category;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::active()->with(['category', 'reviews']);

        // تصفية حسب الفئة
        if ($request->has('category') && $request->category != 'all') {
            $query->byCategory($request->category);
        }

        // البحث
        if ($request->has('search') && !empty($request->search)) {
            $query->search($request->search);
        }

        // تصفية حسب السعر
        if ($request->has('min_price') || $request->has('max_price')) {
            $query->priceRange($request->min_price, $request->max_price);
        }

        // ترتيب النتائج
        $sortBy = $request->get('sort', 'popular');
        switch ($sortBy) {
            case 'name_asc':
                $query->orderBy('name');
                break;
            case 'name_desc':
                $query->orderBy('name', 'desc');
                break;
            case 'price_asc':
                $query->orderBy('price');
                break;
            case 'price_desc':
                $query->orderBy('price', 'desc');
                break;
            case 'newest':
                $query->latest();
                break;
            case 'rating':
                $query->withAvg('reviews', 'rating')
                      ->orderBy('reviews_avg_rating', 'desc');
                break;
            default: // popular
                $query->where('is_popular', true)
                      ->orWhere('is_featured', true)
                      ->orderBy('is_popular', 'desc')
                      ->orderBy('is_featured', 'desc');
                break;
        }

        $products = $query->paginate(12)->withQueryString();

        // جميع الفئات للتصفية
        $categories = Category::active()
            ->withCount('activeProducts')
            ->ordered()
            ->get();

        // المنتجات المميزة للعرض الجانبي
        $featuredProducts = Product::active()
            ->featured()
            ->take(3)
            ->get();

        return view('products.index', compact(
            'products', 
            'categories', 
            'featuredProducts'
        ));
    }

    public function show(Product $product)
    {
        // التأكد من أن المنتج نشط
        if (!$product->is_active) {
            abort(404);
        }

        // تحميل البيانات المرتبطة
        $product->load(['category', 'reviews.user']);

        // منتجات مشابهة
        $relatedProducts = Product::active()
            ->where('category_id', $product->category_id)
            ->where('id', '!=', $product->id)
            ->take(4)
            ->get();

        // منتجات مقترحة
        $suggestedProducts = Product::active()
            ->popular()
            ->where('id', '!=', $product->id)
            ->take(4)
            ->get();

        return view('products.show', compact(
            'product',
            'relatedProducts', 
            'suggestedProducts'
        ));
    }

    public function quickView(Product $product)
    {
        if (!$product->is_active) {
            return response()->json(['error' => 'المنتج غير متاح'], 404);
        }

        $product->load('category');

        return response()->json([
            'product' => [
                'id' => $product->id,
                'name' => $product->name,
                'description' => $product->description,
                'price' => $product->price,
                'original_price' => $product->original_price,
                'discount_percentage' => $product->discount_percentage,
                'main_image' => $product->main_image,
                'all_images' => $product->all_images,
                'category' => $product->category?->name,
                'preparation_time' => $product->preparation_time,
                'calories' => $product->calories,
                'ingredients' => $product->ingredients,
                'customizations' => $product->customizations,
                'average_rating' => $product->average_rating,
                'reviews_count' => $product->reviews_count,
                'formatted_price' => $product->formatPrice()
            ]
        ]);
    }

    public function getByCategory(Category $category)
    {
        $products = Product::active()
            ->byCategory($category->id)
            ->with('category')
            ->orderBy('sort_order')
            ->orderBy('name')
            ->paginate(12);

        $categories = Category::active()
            ->withCount('activeProducts')
            ->ordered()
            ->get();

        return view('products.category', compact(
            'products',
            'category',
            'categories'
        ));
    }

    public function featured()
    {
        $products = Product::active()
            ->featured()
            ->with('category')
            ->paginate(12);

        $categories = Category::active()
            ->withCount('activeProducts')
            ->ordered()
            ->get();

        return view('products.featured', compact('products', 'categories'));
    }

    public function popular()
    {
        $products = Product::active()
            ->popular()
            ->withAvg('reviews', 'rating')
            ->with('category')
            ->paginate(12);

        $categories = Category::active()
            ->withCount('activeProducts')
            ->ordered()
            ->get();

        return view('products.popular', compact('products', 'categories'));
    }

    public function offers()
    {
        $products = Product::active()
            ->whereNotNull('original_price')
            ->whereColumn('price', '<', 'original_price')
            ->with('category')
            ->orderByRaw('((original_price - price) / original_price * 100) DESC')
            ->paginate(12);

        $categories = Category::active()
            ->withCount('activeProducts')
            ->ordered()
            ->get();

        return view('products.offers', compact('products', 'categories'));
    }

    public function search(Request $request)
    {
        $query = $request->get('q');
        
        if (empty($query)) {
            return redirect()->route('products.index');
        }

        $products = Product::active()
            ->search($query)
            ->with('category')
            ->paginate(12);

        $categories = Category::active()
            ->where(function($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                  ->orWhere('name_en', 'like', "%{$query}%");
            })
            ->get();

        return view('products.search', compact('products', 'categories', 'query'));
    }

    public function autocomplete(Request $request)
    {
        $query = $request->get('q');
        
        if (strlen($query) < 2) {
            return response()->json([]);
        }

        $products = Product::active()
            ->search($query)
            ->select('id', 'name', 'price', 'image')
            ->take(5)
            ->get()
            ->map(function($product) {
                return [
                    'id' => $product->id,
                    'name' => $product->name,
                    'price' => number_format($product->price, 0) . ' جنيه',
                    'image' => $product->main_image,
                    'url' => route('products.show', $product)
                ];
            });

        return response()->json($products);
    }

    public function addToFavorites(Request $request, Product $product)
    {
        if (!auth()->check()) {
            return response()->json([
                'success' => false,
                'message' => 'يجب تسجيل الدخول أولاً'
            ], 401);
        }

        $user = auth()->user();
        
        $favorite = $user->favorites()->where('product_id', $product->id)->first();
        
        if ($favorite) {
            $favorite->delete();
            $message = 'تم إزالة المنتج من المفضلة';
            $action = 'removed';
        } else {
            $user->favorites()->create(['product_id' => $product->id]);
            $message = 'تم إضافة المنتج إلى المفضلة';
            $action = 'added';
        }

        return response()->json([
            'success' => true,
            'message' => $message,
            'action' => $action
        ]);
    }

    public function getCustomizationOptions(Product $product)
    {
        if (!$product->canBeCustomized()) {
            return response()->json(['customizations' => []]);
        }

        return response()->json([
            'customizations' => $product->customizations
        ]);
    }
}
