<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Coupon extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'code',
        'type',
        'value',
        'minimum_amount',
        'max_discount_amount',
        'usage_limit',
        'used_count',
        'is_active',
        'starts_at',
        'expires_at',
        'description'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'starts_at' => 'datetime',
        'expires_at' => 'datetime',
        'value' => 'decimal:2',
        'minimum_amount' => 'decimal:2',
        'max_discount_amount' => 'decimal:2'
    ];

    // Types
    const TYPE_PERCENTAGE = 'percentage';
    const TYPE_FIXED = 'fixed';

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeValid($query)
    {
        return $query->where('is_active', true)
                    ->where('starts_at', '<=', now())
                    ->where('expires_at', '>=', now());
    }

    public function scopeByCode($query, $code)
    {
        return $query->where('code', $code);
    }

    // Helper methods
    public function isValid()
    {
        return $this->is_active 
               && $this->starts_at <= now() 
               && $this->expires_at >= now()
               && (!$this->usage_limit || $this->used_count < $this->usage_limit);
    }

    public function canBeUsed()
    {
        return $this->isValid();
    }

    public function getRemainingUsesAttribute()
    {
        if (!$this->usage_limit) {
            return null; // Unlimited
        }
        
        return max(0, $this->usage_limit - $this->used_count);
    }

    public function getDiscountText()
    {
        if ($this->type === self::TYPE_PERCENTAGE) {
            $text = $this->value . '% خصم';
            if ($this->max_discount_amount) {
                $text .= ' (حد أقصى ' . number_format($this->max_discount_amount, 0) . ' جنيه)';
            }
            return $text;
        } else {
            return number_format($this->value, 0) . ' جنيه خصم';
        }
    }

    public function calculateDiscount($amount)
    {
        if (!$this->canBeUsed()) {
            return 0;
        }

        if ($this->minimum_amount && $amount < $this->minimum_amount) {
            return 0;
        }

        $discount = 0;
        
        if ($this->type === self::TYPE_PERCENTAGE) {
            $discount = ($amount * $this->value) / 100;
            if ($this->max_discount_amount) {
                $discount = min($discount, $this->max_discount_amount);
            }
        } else {
            $discount = $this->value;
        }

        return min($discount, $amount);
    }

    public function incrementUsage()
    {
        $this->increment('used_count');
    }

    public function getTypeTextAttribute()
    {
        return $this->type === self::TYPE_PERCENTAGE ? 'نسبة مئوية' : 'قيمة ثابتة';
    }

    public function getStatusTextAttribute()
    {
        if (!$this->is_active) {
            return 'غير نشط';
        }
        
        if ($this->starts_at > now()) {
            return 'لم يبدأ بعد';
        }
        
        if ($this->expires_at < now()) {
            return 'منتهي الصلاحية';
        }
        
        if ($this->usage_limit && $this->used_count >= $this->usage_limit) {
            return 'تم استنفاد الاستخدامات';
        }
        
        return 'نشط';
    }
}
