<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Product extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'name',
        'name_en',
        'description',
        'description_en',
        'price',
        'original_price',
        'category_id',
        'image',
        'images',
        'is_active',
        'is_featured',
        'is_popular',
        'preparation_time',
        'calories',
        'ingredients',
        'allergens',
        'customizations',
        'sort_order'
    ];

    protected $casts = [
        'images' => 'array',
        'ingredients' => 'array',
        'allergens' => 'array',
        'customizations' => 'array',
        'is_active' => 'boolean',
        'is_featured' => 'boolean',
        'is_popular' => 'boolean',
        'price' => 'decimal:2',
        'original_price' => 'decimal:2'
    ];

    // العلاقات
    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    public function orderItems()
    {
        return $this->hasMany(OrderItem::class);
    }

    public function reviews()
    {
        return $this->hasMany(Review::class);
    }

    public function favorites()
    {
        return $this->hasMany(Favorite::class);
    }

    // الطرق المساعدة
    public function getDiscountPercentageAttribute()
    {
        if ($this->original_price && $this->original_price > $this->price) {
            return round((($this->original_price - $this->price) / $this->original_price) * 100);
        }
        return 0;
    }

    public function getAverageRatingAttribute()
    {
        return $this->reviews()->avg('rating') ?? 0;
    }

    public function getReviewsCountAttribute()
    {
        return $this->reviews()->count();
    }

    public function getSavingsAttribute()
    {
        if ($this->original_price && $this->original_price > $this->price) {
            return $this->original_price - $this->price;
        }
        return 0;
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    public function scopePopular($query)
    {
        return $query->where('is_popular', true);
    }

    public function scopeByCategory($query, $categoryId)
    {
        return $query->where('category_id', $categoryId);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('name', 'like', "%{$search}%")
              ->orWhere('name_en', 'like', "%{$search}%")
              ->orWhere('description', 'like', "%{$search}%")
              ->orWhere('description_en', 'like', "%{$search}%");
        });
    }

    public function scopePriceRange($query, $min = null, $max = null)
    {
        if ($min) {
            $query->where('price', '>=', $min);
        }
        if ($max) {
            $query->where('price', '<=', $max);
        }
        return $query;
    }

    // Helper methods
    public function getMainImageAttribute()
    {
        if ($this->image) {
            return asset('storage/products/' . $this->image);
        }
        return asset('images/default-product.jpg');
    }

    public function getAllImagesAttribute()
    {
        $images = [];
        
        // Main image
        if ($this->image) {
            $images[] = asset('storage/products/' . $this->image);
        }
        
        // Additional images
        if ($this->images && is_array($this->images)) {
            foreach ($this->images as $img) {
                $images[] = asset('storage/products/' . $img);
            }
        }
        
        return $images ?: [asset('images/default-product.jpg')];
    }

    public function formatPrice($includeDiscount = true)
    {
        $price = number_format($this->price, 0) . ' جنيه';
        
        if ($includeDiscount && $this->original_price && $this->original_price > $this->price) {
            $originalPrice = number_format($this->original_price, 0) . ' جنيه';
            return [
                'current' => $price,
                'original' => $originalPrice,
                'discount' => $this->discount_percentage . '%'
            ];
        }
        
        return ['current' => $price];
    }

    public function canBeCustomized()
    {
        return !empty($this->customizations) && is_array($this->customizations);
    }

    public function getCustomizationOptions($type = null)
    {
        if (!$this->canBeCustomized()) {
            return [];
        }

        if ($type) {
            return $this->customizations[$type] ?? [];
        }

        return $this->customizations;
    }
}
