const CACHE_NAME = 'burger-on-fire-laravel-v1.0.0';
const urlsToCache = [
  '/',
  '/css/app.css',
  '/js/app.js',
  '/manifest.json',
  // Laravel static assets
  '/images/logo.png',
  '/images/hero-burger-1.jpg',
  '/images/hero-burger-2.jpg', 
  '/images/hero-burger-3.jpg',
  // Add other important static assets
];

// تثبيت Service Worker
self.addEventListener('install', (event) => {
  console.log('[SW] Installing...');
  event.waitUntil(
    caches.open(CACHE_NAME)
      .then((cache) => {
        console.log('[SW] Caching app shell');
        return cache.addAll(urlsToCache);
      })
      .catch((error) => {
        console.error('[SW] Install failed:', error);
      })
  );
  // تفعيل Service Worker الجديد فوراً
  self.skipWaiting();
});

// تفعيل Service Worker
self.addEventListener('activate', (event) => {
  console.log('[SW] Activating...');
  event.waitUntil(
    caches.keys().then((cacheNames) => {
      return Promise.all(
        cacheNames.map((cacheName) => {
          if (cacheName !== CACHE_NAME) {
            console.log('[SW] Deleting old cache:', cacheName);
            return caches.delete(cacheName);
          }
        })
      );
    })
  );
  // السيطرة على جميع النوافذ المفتوحة
  self.clients.claim();
});

// اعتراض طلبات الشبكة
self.addEventListener('fetch', (event) => {
  // تجاهل طلبات non-GET
  if (event.request.method !== 'GET') {
    return;
  }

  // تجاهل طلبات chrome-extension و similar
  if (!event.request.url.startsWith('http')) {
    return;
  }

  // تجاهل طلبات API للحصول على بيانات طازجة
  if (event.request.url.includes('/api/')) {
    return;
  }

  event.respondWith(
    caches.match(event.request)
      .then((response) => {
        // إرجاع النسخة المحفوظة إذا وجدت
        if (response) {
          console.log('[SW] Serving from cache:', event.request.url);
          return response;
        }

        console.log('[SW] Fetching from network:', event.request.url);
        return fetch(event.request).then((response) => {
          // التحقق من صحة الاستجابة
          if (!response || response.status !== 200 || response.type !== 'basic') {
            return response;
          }

          // استنساخ الاستجابة
          const responseToCache = response.clone();

          caches.open(CACHE_NAME)
            .then((cache) => {
              // حفظ الملفات الثابتة فقط
              if (event.request.url.match(/\.(css|js|png|jpg|jpeg|gif|ico|svg|woff|woff2|ttf|eot)$/)) {
                cache.put(event.request, responseToCache);
              }
            });

          return response;
        });
      }
    ).catch((error) => {
      console.error('[SW] Fetch failed:', error);
      
      // في حالة عدم توفر الاتصال، إرجاع صفحة offline للصفحات HTML
      if (event.request.destination === 'document') {
        return caches.match('/offline.html');
      }
      
      // للصور، إرجاع صورة placeholder
      if (event.request.destination === 'image') {
        return new Response('<svg xmlns="http://www.w3.org/2000/svg" width="200" height="200" viewBox="0 0 200 200"><rect width="200" height="200" fill="#f3f4f6"/><text x="50%" y="50%" text-anchor="middle" dy="0.35em" font-family="Arial" font-size="14" fill="#6b7280">غير متاح</text></svg>', {
          headers: { 'Content-Type': 'image/svg+xml' }
        });
      }
    })
  );
});

// معالجة رسائل Push
self.addEventListener('push', (event) => {
  console.log('[SW] Push message received');
  
  let data = {};
  if (event.data) {
    data = event.data.json();
  }

  const title = data.title || 'أون فاير 🔥';
  const options = {
    body: data.body || 'لديك إشعار جديد',
    icon: '/images/icon-192x192.png',
    badge: '/images/badge-72x72.png',
    image: data.image,
    vibrate: [200, 100, 200],
    data: {
      dateOfArrival: Date.now(),
      primaryKey: data.primaryKey || 1,
      url: data.url || '/'
    },
    actions: [
      {
        action: 'explore',
        title: '🔍 عرض التفاصيل',
        icon: '/images/icon-explore.png'
      },
      {
        action: 'close',
        title: '❌ إغلاق',
        icon: '/images/icon-close.png'
      }
    ],
    tag: data.tag || 'general',
    renotify: true,
    requireInteraction: false,
    silent: false
  };

  event.waitUntil(
    self.registration.showNotification(title, options)
  );
});

// معالجة النقر على الإشعارات
self.addEventListener('notificationclick', (event) => {
  console.log('[SW] Notification click received.');

  event.notification.close();

  if (event.action === 'explore') {
    // فتح رابط محدد
    event.waitUntil(
      clients.openWindow(event.notification.data.url)
    );
  } else if (event.action === 'close') {
    // إغلاق الإشعار فقط
    console.log('[SW] Notification dismissed');
  } else {
    // النقر العادي على الإشعار
    event.waitUntil(
      clients.matchAll({ type: 'window' }).then((clientList) => {
        for (const client of clientList) {
          if (client.url === self.location.origin + '/' && 'focus' in client) {
            return client.focus();
          }
        }
        if (clients.openWindow) {
          return clients.openWindow('/');
        }
      })
    );
  }
});

// معالجة إغلاق الإشعارات
self.addEventListener('notificationclose', (event) => {
  console.log('[SW] Notification closed');
  
  // يمكن إضافة تحليلات هنا
  // analytics.track('notification-closed', { tag: event.notification.tag });
});

// معالجة المزامنة في الخلفية
self.addEventListener('sync', (event) => {
  console.log('[SW] Background sync triggered:', event.tag);
  
  if (event.tag === 'background-sync') {
    event.waitUntil(doBackgroundSync());
  } else if (event.tag === 'cart-sync') {
    event.waitUntil(syncCart());
  } else if (event.tag === 'order-sync') {
    event.waitUntil(syncOrder());
  }
});

async function doBackgroundSync() {
  try {
    console.log('[SW] Performing background sync');
    
    // مزامنة البيانات المعلقة
    const pendingRequests = await getPendingRequests();
    
    for (const request of pendingRequests) {
      try {
        await fetch(request.url, request.options);
        await clearPendingRequest(request.id);
        console.log('[SW] Synced request:', request.url);
      } catch (error) {
        console.error('[SW] Failed to sync request:', request.url, error);
      }
    }
    
    return Promise.resolve();
  } catch (error) {
    console.error('[SW] Background sync failed:', error);
    return Promise.reject(error);
  }
}

async function syncCart() {
  try {
    // مزامنة سلة التسوق
    const cart = await getLocalCart();
    if (cart && cart.items.length > 0) {
      await fetch('/api/cart/sync', {
        method: 'POST',
        body: JSON.stringify(cart),
        headers: {
          'Content-Type': 'application/json'
        }
      });
      console.log('[SW] Cart synced successfully');
    }
  } catch (error) {
    console.error('[SW] Cart sync failed:', error);
  }
}

async function syncOrder() {
  try {
    // مزامنة الطلبات المعلقة
    const pendingOrders = await getPendingOrders();
    
    for (const order of pendingOrders) {
      try {
        const response = await fetch('/api/orders', {
          method: 'POST',
          body: JSON.stringify(order),
          headers: {
            'Content-Type': 'application/json'
          }
        });
        
        if (response.ok) {
          await clearPendingOrder(order.id);
          console.log('[SW] Order synced:', order.id);
        }
      } catch (error) {
        console.error('[SW] Failed to sync order:', order.id, error);
      }
    }
  } catch (error) {
    console.error('[SW] Order sync failed:', error);
  }
}

// وظائف مساعدة للتخزين المحلي
async function getPendingRequests() {
  return []; // يمكن استخدام IndexedDB
}

async function clearPendingRequest(id) {
  // مسح الطلب المعلق
}

async function getLocalCart() {
  return null; // يمكن استخدام IndexedDB
}

async function getPendingOrders() {
  return []; // يمكن استخدام IndexedDB
}

async function clearPendingOrder(id) {
  // مسح الطلب المعلق
}

// معالجة رسائل من التطبيق الرئيسي
self.addEventListener('message', (event) => {
  console.log('[SW] Message received:', event.data);
  
  if (event.data && event.data.type === 'SKIP_WAITING') {
    self.skipWaiting();
  } else if (event.data && event.data.type === 'UPDATE_CACHE') {
    // تحديث ملفات معينة في الذاكرة التخزينية
    updateSpecificFiles(event.data.files);
  } else if (event.data && event.data.type === 'CLEAR_CACHE') {
    // مسح الذاكرة التخزينية
    clearAllCaches();
  }
  
  // إرسال رد
  event.ports[0]?.postMessage({
    type: 'SW_RESPONSE',
    success: true
  });
});

async function updateSpecificFiles(files) {
  try {
    const cache = await caches.open(CACHE_NAME);
    await cache.addAll(files);
    console.log('[SW] Updated cache for files:', files);
  } catch (error) {
    console.error('[SW] Failed to update cache:', error);
  }
}

async function clearAllCaches() {
  try {
    const cacheNames = await caches.keys();
    await Promise.all(
      cacheNames.map(cacheName => caches.delete(cacheName))
    );
    console.log('[SW] All caches cleared');
  } catch (error) {
    console.error('[SW] Failed to clear caches:', error);
  }
}

// التعامل مع الأخطاء غير المعالجة
self.addEventListener('error', (event) => {
  console.error('[SW] Error occurred:', event.error);
});

self.addEventListener('unhandledrejection', (event) => {
  console.error('[SW] Unhandled promise rejection:', event.reason);
});

console.log('[SW] 🔥 Burger On Fire PWA Service Worker loaded successfully!');
